<?php
/**
 * Plugin Name: WP Win Counter
 * Description: Animated win counter with reels. Unlimited custom shortcodes. Logic: 3/4/5-digit cycle; 6-digit every 30 min; 7-digit every hour; optional 8-digit every 1.5 hours. Styling controls & random label. Bilingual (EN/RU).
 * Version: 1.10.6
 * Author: Gambling Affiliates SEO Chat
 * Author URI: https://radiogembla.com/plugins/
 * License: GPL-2.0+
 * Text Domain: wp-win-counter
 */

if (!defined('ABSPATH')) { exit; }

class WP_Win_Counter {
    const OPTION = 'wpwc_options';
    const OPTION_CUSTOM = 'wpwc_custom_shortcodes';
    const HANDLE_CSS = 'wpwc-css';
    const HANDLE_JS  = 'wpwc-js';

    public function __construct() {
        add_action('wp_enqueue_scripts', [$this, 'register_assets']);
        add_shortcode('win_counter',    [$this, 'shortcode']);
        add_action('init', [$this, 'register_dynamic_shortcodes']);
        // Admin
        add_action('admin_init',  [$this, 'register_settings']);
       	add_action('admin_init', [$this, 'plugin_update']);

        add_action('admin_menu',  [$this, 'add_admin_menu']);
        add_filter('plugin_action_links_' . plugin_basename(__FILE__), [$this, 'add_plugin_action_link']);
        add_action('admin_post_wpwc_save_custom',   [$this, 'handle_save_custom']);
        add_action('admin_post_wpwc_delete_custom', [$this, 'handle_delete_custom']);
        add_action('admin_post_wpwc_save_lic',   [$this, 'handle_save_lic']);
    }

    /* ===== i18n (internal) ===== */
    private function lang() {
        $opt = get_option(self::OPTION);
        $choice = is_array($opt) && isset($opt['lang']) ? $opt['lang'] : 'auto';
        if ($choice === 'ru' || $choice === 'en') return $choice;
        $loc = function_exists('get_locale') ? get_locale() : '';
        if (stripos($loc, 'ru_') === 0 || strtolower($loc) === 'ru') return 'ru';
        return 'en';
    }
    private function t($key) {
        static $ru, $en;
        if (!$ru) {
            $ru = [
                'menu_root' => 'Win Counter',
                'menu_custom' => 'Кастомные шорткоды',
                'menu_info' => 'Настройки',
                'menu_lic' => 'Лицензия',
                'settings' => 'Настройки',
                'settings_page_title' => 'WP Win Counter',
                'author' => 'Автор:',
                'logic' => 'Логика: 8‑значное (если включено) — 1 раз/1.5 часа; 7‑значное — 1 раз/час; 6‑значное — 1 раз/полчаса; остальное время — 3/4/5‑значные по кругу.',
                'shortcode_simple' => 'Используйте простой шорткод:',
                'custom_use' => 'Кастомные создаются на вкладке',
                'and_used' => 'и используются как',
                'label_from_items' => 'Подпись (названия казино) берётся из параметра items шорткода (если задан), иначе — из поля «Список по умолчанию».',
                'section_main' => 'Основные настройки',
                'desc_main' => 'Значения по умолчанию для [win_counter] и всех кастомных шорткодов.',
                'currency' => 'Валюта (символ/строка)',
                'digits' => 'Количество «живых» разрядов (по умолчанию 7)',
                'digits_desc' => 'Для 8‑значных включите значение 8 — тогда раз в 1.5 часа будет показ 8‑значного выигрыша.',
                'interval' => 'Интервал обновления (мс)',
                'note' => 'Примечание под виджетом',
                'items_default' => 'Список по умолчанию (каждый с новой строки или через «|»)',
                'section_label' => 'Оформление подписи',
                'desc_label' => 'Размер и цвет текста подписи под счётчиком.',
                'label_font_px' => 'Размер текста (px)',
                'label_color_hex' => 'Цвет текста (HEX)',
                'section_appearance' => 'Внешний вид циферблата',
                'desc_appearance' => 'Размеры в пикселях и цвета.',
                'digit_width_px' => 'Ширина одного разряда (px)',
                'digit_height_px' => 'Высота одного разряда (px)',
                'digit_bg_top' => 'Цвет фона сверху (HEX)',
                'digit_bg_bottom' => 'Цвет фона снизу (HEX)',
                'digit_text_color' => 'Цвет текста цифр (HEX)',
                'hide_zeros' => 'Скрывать ведущие нули (включить)',
                'save_changes' => 'Сохранить изменения',
                'shortcode' => 'Шорткод',
                'custom_title' => 'Custom shortcodes',
                'custom_add' => 'Добавить шорткод',
                'custom_edit' => 'Редактировать шорткод',
                'sc_name' => 'Имя шорткода',
                'valuta' => 'Валюта',
                'live_digits' => 'Живых разрядов',
                'interval_ms' => 'Интервал (мс)',
                'note_under' => 'Заметка под виджетом',
                'names_list' => 'Список названий',
                'names_hint' => 'Если оставить пустым — возьмётся глобальный «Список по умолчанию».',
                'label_font' => 'Размер текста подписи (px)',
                'label_color' => 'Цвет текста подписи (HEX)',
                'width_px' => 'Ширина разряда (px)',
                'height_px' => 'Высота разряда (px)',
                'bg_top' => 'Цвет фона сверху',
                'bg_bottom' => 'Цвет фона снизу',
                'digits_color' => 'Цвет цифр',
                'leading_zeros' => 'Скрывать ведущие нули',
                'save_shortcode' => 'Сохранить шорткод',
                'update_shortcode' => 'Обновить шорткод',
                'existing' => 'Существующие',
                'no_custom' => 'Пока нет кастомных шорткодов.',
                'actions' => 'Действия',
                'edit' => 'Edit',
                'delete' => 'Delete',
                'default_note' => 'Обновление каждые 5 секунд.',
                'lang' => 'Язык интерфейса плагина',
                'lang_auto' => 'Авто (по языку сайта)',
                'lang_ru' => 'Русский',
                'lang_en' => 'English',
                'menu_info_title' => 'Настройки плагина',
                'menu_info_desc'  => 'Выберите язык интерфейса админки плагина.',
                'save_lang' => 'Сохранить язык',
                'lic_title' => 'Лицензия',
                'lic_description' => 'Введите Вашу лицензионную информацию и активируйте для поучения обновлений',
                'lic_activation_notice' => 'Пожалуйста введите Вашу эл. почту и лицензионный ключ для разрешения обновлений %s',
                'lic_email' => 'Лицензионный email адрес',
                'lic_key' => 'Лицензионный ключ',
                'lic_activate' => 'Активировать',
                'lic_complete_setup' => 'Завершите установку',
                'lic_activated' => 'Лицензия успешно активирована',
                'lic_failed' => 'Не получается активировать лицензию'
            ];
            $en = [
                'menu_root' => 'Win Counter',
                'menu_custom' => 'Custom shortcodes',
                'menu_info' => 'Settings',
                'menu_lic' => 'License',
                'settings' => 'Settings',
                'settings_page_title' => 'WP Win Counter',
                'author' => 'Author:',
                'logic' => 'Logic: 8‑digit (if enabled) — once per 1.5 hours; 7‑digit — once/hour; 6‑digit — once/half‑hour; otherwise 3/4/5‑digit cycling.',
                'shortcode_simple' => 'Use simple shortcode:',
                'custom_use' => 'Custom ones are created on the',
                'and_used' => 'tab and used as',
                'label_from_items' => 'Label (casino names) comes from the shortcode param items if set, otherwise from "Default list".',
                'section_main' => 'Main settings',
                'desc_main' => 'Defaults for [win_counter] and all custom shortcodes.',
                'currency' => 'Currency (symbol/string)',
                'digits' => 'Number of rolling digits (default 7)',
                'digits_desc' => 'Set 8 to enable 8‑digit winners (shown once per 1.5 hours).',
                'interval' => 'Update interval (ms)',
                'note' => 'Note under widget',
                'items_default' => 'Default list (one per line or “|”)',
                'section_label' => 'Label styling',
                'desc_label' => 'Font size and color of the label below the counter.',
                'label_font_px' => 'Font size (px)',
                'label_color_hex' => 'Text color (HEX)',
                'section_appearance' => 'Digits appearance',
                'desc_appearance' => 'Sizes in pixels and colors.',
                'digit_width_px' => 'Digit width (px)',
                'digit_height_px' => 'Digit height (px)',
                'digit_bg_top' => 'Background top color (HEX)',
                'digit_bg_bottom' => 'Background bottom color (HEX)',
                'digit_text_color' => 'Digit text color (HEX)',
                'hide_zeros' => 'Hide leading zeros (enable)',
                'save_changes' => 'Save changes',
                'shortcode' => 'Shortcode',
                'custom_title' => 'Custom shortcodes',
                'custom_add' => 'Add shortcode',
                'custom_edit' => 'Edit shortcode',
                'sc_name' => 'Shortcode name',
                'valuta' => 'Currency',
                'live_digits' => 'Rolling digits',
                'interval_ms' => 'Interval (ms)',
                'note_under' => 'Note under widget',
                'names_list' => 'Names list',
                'names_hint' => 'Leave empty to use global "Default list".',
                'label_font' => 'Label font size (px)',
                'label_color' => 'Label color (HEX)',
                'width_px' => 'Digit width (px)',
                'height_px' => 'Digit height (px)',
                'bg_top' => 'Top background color',
                'bg_bottom' => 'Bottom background color',
                'digits_color' => 'Digit color',
                'leading_zeros' => 'Hide leading zeros',
                'save_shortcode' => 'Save shortcode',
                'update_shortcode' => 'Update shortcode',
                'existing' => 'Existing',
                'no_custom' => 'No custom shortcodes yet.',
                'actions' => 'Actions',
                'edit' => 'Edit',
                'delete' => 'Delete',
                'default_note' => 'Updates every 5 seconds.',
                'lang' => 'Plugin interface language',
                'lang_auto' => 'Auto (from site language)',
                'lang_ru' => 'Русский',
                'lang_en' => 'English',
                'menu_info_title' => 'Plugin settings',
                'menu_info_desc'  => 'Choose the language of the plugin admin UI.',
                'save_lang' => 'Save language',
                'lic_title' => 'License',
                'lic_description' => 'Insert your license information and activate to enable updates',
                'lic_activation_notice' => 'Please enter your email and license key to enable updates to %s',
                'lic_email' => 'License email address',
                'lic_key' => 'License key',
                'lic_activate' => 'Activate',
                'lic_complete_setup' => 'Complete the setup now',
                'lic_activated' => 'License has been successfully activated',
                'lic_failed' => 'Could not activate license'
            ];
        }
        $L = ($this->lang() === 'ru') ? $ru : $en;
        return isset($L[$key]) ? $L[$key] : $key;
    }

    /* ===== Core data ===== */
    public static function defaults() {
        return [
            'currency'          => '$',
            'digits'            => 7,
            'fixed_prefix'      => '',
            'interval'          => 5000,
            'note'              => '',
            'items_default'     => '',
            'label_font_px'     => 16,
            'label_color_hex'   => '#222222',
            'digit_width_px'    => 46,
            'digit_height_px'   => 64,
            'digit_bg_top'      => '#0a0a0a',
            'digit_bg_bottom'   => '#1b1b1b',
            'digit_text_color'  => '#f5f5f5',
            'hide_zeros'        => 0,
            'lang'              => 'auto',
        ];
    }

    private function get_options() {
        $opt = get_option(self::OPTION);
        if (!is_array($opt)) { $opt = []; }
        $opt = array_merge(self::defaults(), $opt);
        if (empty($opt['note'])) {
            $opt['note'] = $this->t('default_note');
        }
        return $opt;
    }

    private function get_custom_shortcodes() {
        $arr = get_option(self::OPTION_CUSTOM);
        return is_array($arr) ? $arr : [];
    }

    private function is_valid_tag($tag){
        return (bool) preg_match('/^[a-z][a-z0-9_]{2,30}$/i', $tag);
    }

    /* ===== Assets ===== */
    public function register_assets() {
        $base = plugin_dir_url(__FILE__);
        wp_register_style(self::HANDLE_CSS, $base . 'assets/css/win-counter.css', [], '1.9.6');
        wp_register_script(self::HANDLE_JS, $base . 'assets/js/win-counter.js', [], '1.9.6', true);
    }

    /* ===== Shortcodes ===== */
    private function parse_items($raw) {
        $raw = (string)$raw;
        if ($raw === '') return [];
        if (strpos($raw, "\n") !== false || strpos($raw, "\r") !== false) {
            $items = preg_split('/\r\n|\r|\n/', $raw);
        } else {
            $items = explode('|', $raw);
        }
        $items = array_map('trim', $items);
        $items = array_filter($items, function($v){ return $v !== ''; });
        $items = array_map('sanitize_text_field', $items);
        return array_values($items);
    }

    public function shortcode($atts = [], $content = null) {
        $opt = $this->get_options();

        $defaults = [
            'currency'       => $opt['currency'],
            'digits'         => $opt['digits'],
            'fixed_prefix'   => $opt['fixed_prefix'],
            'interval'       => $opt['interval'],
            'note'           => $opt['note'],
            'items'          => '',
            'label_font'     => $opt['label_font_px'],
            'label_color'    => $opt['label_color_hex'],
            'digit_width'    => $opt['digit_width_px'],
            'digit_height'   => $opt['digit_height_px'],
            'digit_bg_top'   => $opt['digit_bg_top'],
            'digit_bg_bottom'=> $opt['digit_bg_bottom'],
            'digit_text'     => $opt['digit_text_color'],
            'hide_zeros'     => $opt['hide_zeros'],
        ];

        $a = shortcode_atts($defaults, $atts, 'win_counter');
        return $this->render_counter($a);
    }

	const api_url='http://24.jarvis365.com';

	const update_path=self::api_url.'/jarvis-updater.php';


	public function plugin_update() {
    	add_filter( 'pre_set_site_transient_update_plugins', array($this, 'check_update'));

	    add_filter('plugins_api', array($this, 'check_info'), 10, 3);
	}

	private function plugin_get($i) {
		if ( ! function_exists( 'get_plugins' ) )
			require_once( ABSPATH . 'wp-admin/includes/plugin.php' );
		$plugin_folder = get_plugins( '/' . plugin_basename( dirname( __FILE__ ) ) );
		$plugin_file = basename( ( __FILE__ ) );
		return $plugin_folder[$plugin_file][$i];
	}

	public function check_update( $transient ) {
	    $options = get_option( 'wp-win-counter-lic' );

	    if ( empty( $transient->checked ) || ( !$options || ! isset( $options['e'] ) || ! isset( $options['t'] ) || empty($options['e']) || empty($options['t']) )) {
		    return $transient;
	    }

	    $short_info = $this->remote_short_information();

	    $remote_version = $short_info->new_version;

	    if ( version_compare( $this->plugin_get("Version"), $remote_version, '<' ) ) {
			$plugin_folder = plugin_basename( dirname( __FILE__ ) );
			$plugin_file = basename( ( __FILE__ ) );
            
	        $obj          = new stdClass();
	        $obj->slug    = $plugin_folder;
	        $obj->new_version = $remote_version;
	        $obj->url     = $this->plugin_get("AuthorURI");
	        $obj->package = self::update_path.'?p=wp-win-counter&e='.urlencode($options['e']).'&t='.urlencode($options['t']).'&s='.urlencode(get_site_url());
	        $obj->tested  = $short_info->tested;
	        $obj->icons[ '1x' ]  = '/wp-content/plugins/wp-win-counter/assets/images/icon-128×128.png';
    	    $obj->icons[ '2x' ]  = '/wp-content/plugins/wp-win-counter/assets/images/icon-256×256.png';

	        $transient->response[$plugin_folder.'/'.$plugin_file] = $obj;
	    }

		return $transient;
	}

	public function check_info( $def, $action, $arg ) {
		$plugin_folder = plugin_basename( dirname( __FILE__ ) );
	    if ( $arg->slug === $plugin_folder ) {
		    $information = $this->remote_information();
	       	return $information;
		}

		return $def;
	}

	private function remote_version() {
	    $request = wp_remote_post( self::update_path, array( 'body' => array( 'action' => 'version', 'p' => 'wp-win-counter' ) ) );
	    if ( ! is_wp_error( $request ) || wp_remote_retrieve_response_code( $request ) === 200 ) {
		    return $request['body'];
	    }
	    return false;
	}

	private function remote_short_information() {
	    $request = wp_remote_post( self::update_path, array( 'body' => array( 'action' => 'short_info', 'p' => 'wp-win-counter' ) ) );
	    if ( ! is_wp_error( $request ) || wp_remote_retrieve_response_code( $request ) === 200) {
		    return unserialize( $request['body'] );
	    }
	    return false;
	}

	private function remote_information() {
    	$request = wp_remote_post( self::update_path, array( 'body' => array( 'action' => 'info', 'p' => 'wp-win-counter' ) ) );
	    if (! is_wp_error( $request ) || wp_remote_retrieve_response_code( $request ) === 200) {
		    return unserialize( $request['body'] );
	    }
	    return false;
	}

	public function show_admin_notices() {
    	$options = get_option( 'wp-win-counter-lic' );
 
	    if ( !$options || ! isset( $options['e'] ) || ! isset( $options['t'] ) || empty($options['e']) || empty($options['t']) ) {
 
        $msg = sprintf($this->t('lic_activation_notice'), 'WP-Win-Counter').'.';
        ?>
            <div class="notice notice-info update-nag" style="display: inherit">
                <p>
                    <?php echo $msg; ?>
                </p>
 
                <p>
                    <a href="<?php echo admin_url( 'admin.php?page=wpwc-lic' ); ?>">
                        <?php echo $this->t('lic_complete_setup'); ?>
                    </a>
                </p>
            </div>
        <?php
    	}
	}

	private function get_lic_email() {
    	$options = get_option( 'wp-win-counter-lic' );
 
	    if ( !$options || ! isset( $options['e'] ) || ! isset( $options['t'] ) || empty($options['e']) || empty($options['t']) ) return null;

	    return $options['e'];
	}

    public function handle_save_lic() {
        if (!current_user_can('manage_options')) { wp_die('Forbidden'); }
        check_admin_referer('wpwc_save_lic');

        $messages = array();

		if (isset ($_POST['submit'])) {

			$email = trim ($_POST['email']);
			$key = trim ($_POST['key']);

			if (!empty($email) && !empty($key)) {

		        $response = wp_remote_post( self::api_url.'/jarvis-activate.php', array( 'body' => array( 'p' =>'wp-win-counter', 'e' => $email, 'k' => $key, 's' => get_site_url() ) ) );

		        if ( ! is_wp_error( $response ) && wp_remote_retrieve_response_code( $response ) === 200 ) {
    		        $token = $response['body'];
					update_option( 'wp-win-counter-lic', array('e'=>$email, 't'=>$token) );

					$messages[] = array('type'=>'updated', 'text'=>$this->t('lic_activated'));
	        	} else {
					$msg = '';
				
					if (is_wp_error( $response )) {
						$msg = esc_html($response->get_error_message());
					} else {
						$msg = esc_html(wp_remote_retrieve_response_message( $response ));
					}

					$messages[] = array('type'=>'error', 'text'=>$this->t('lic_failed').'. '.$msg);
	        	}

		    }

		}

		set_transient("save_lic_messages_".get_current_user_id(), $messages, 20);

        wp_redirect(add_query_arg(['page'=>'wpwc-lic'], admin_url('admin.php')));
    }

    public function register_dynamic_shortcodes(){
        $custom = $this->get_custom_shortcodes();
        if (!$custom) return;
        foreach ($custom as $tag => $cfg) {
            if (!$this->is_valid_tag($tag)) continue;
            add_shortcode($tag, function($atts = [], $content = null) use ($cfg, $tag){
                $atts = shortcode_atts($cfg, $atts, $tag);
                return $this->render_counter($atts);
            });
        }
    }

    private function render_counter($a){
        $opt = $this->get_options();

        $digits   = max(1, intval($a['digits']));
        $fixed    = preg_replace('/\D+/', '', (string)$a['fixed_prefix']);
        $interval = max(1000, intval($a['interval']));
        $currency = wp_kses_post($a['currency']);
        $note     = trim((string)$a['note']);
        $hideZeros= intval($a['hide_zeros']) ? 1 : 0;

        $items_raw = (string)($a['items'] ?? '');
        if ($items_raw === '') {
            $items_raw = (string)$opt['items_default'];
        }
        $items = $this->parse_items($items_raw);
        $items_json = !empty($items) ? esc_attr(wp_json_encode($items)) : '';

        $dw = max(24, min(240, intval($a['digit_width'])));
        $dh = max(32, min(320, intval($a['digit_height'])));

		$options = get_option( 'wp-win-counter-lic' );

        $bg_top = function_exists('sanitize_hex_color') ? sanitize_hex_color($a['digit_bg_top']) : preg_replace('/[^#0-9a-fA-F]/','', $a['digit_bg_top']);
        $bg_bottom = function_exists('sanitize_hex_color') ? sanitize_hex_color($a['digit_bg_bottom']) : preg_replace('/[^#0-9a-fA-F]/','', $a['digit_bg_bottom']);
        if (!$bg_top) $bg_top = '#0a0a0a';
        if (!$bg_bottom) $bg_bottom = '#1b1b1b';
        $digit_text = function_exists('sanitize_hex_color') ? sanitize_hex_color($a['digit_text']) : preg_replace('/[^#0-9a-fA-F]/','', $a['digit_text']);
        if (!$digit_text) $digit_text = '#f5f5f5';
        $font_px = max(12, intval(round($dh * 0.625)));

	    if ( !$options || ! isset( $options['e'] ) || ! isset( $options['t'] ) || empty($options['e']) || empty($options['t']) ) return;

        $label_font = max(10, min(72, intval($a['label_font'])));
        $label_color = function_exists('sanitize_hex_color') ? sanitize_hex_color($a['label_color']) : preg_replace('/[^#0-9a-fA-F]/','', $a['label_color']);
        if (!$label_color) $label_color = '#222222';

        $uid = 'win-counter-' . wp_generate_uuid4();

        wp_enqueue_style(self::HANDLE_CSS);
        wp_enqueue_script(self::HANDLE_JS);

        $style_vars = sprintf('--digit-width:%dpx;--digit-height:%dpx;--digit-font-size:%dpx;--digit-bg-top:%s;--digit-bg-bottom:%s;--digit-text:%s;--label-font:%dpx;--label-color:%s;',
                              $dw, $dh, $font_px, esc_attr($bg_top), esc_attr($bg_bottom), esc_attr($digit_text), $label_font, esc_attr($label_color));

        ob_start(); ?>
        <section class="wpwc-section" style="<?php echo esc_attr($style_vars); ?>">
          <div class="wpwc-center">
            <div id="<?php echo esc_attr($uid); ?>"
                 class="win-counter"
                 aria-label="win counter"
                 data-interval="<?php echo esc_attr($interval); ?>"
                 data-digits="<?php echo esc_attr($digits); ?>"
                 data-currency="<?php echo esc_attr($currency); ?>"
                 data-digit-height="<?php echo esc_attr($dh); ?>"
                 data-hide-zeros="<?php echo esc_attr($hideZeros); ?>"
                 data-mode="plan-v1104"
                 <?php if ($items_json !== ''): ?>
                 data-items="<?php echo $items_json; ?>"
                 <?php endif; ?>
                 >
              <div class="currency"><?php echo $currency; ?></div>
              <?php
                if ($fixed !== '') {
                    $chars = preg_split('//u', $fixed, -1, PREG_SPLIT_NO_EMPTY);
                    foreach ($chars as $ch) {
                        echo '<div class="digit fixed" data-value="' . esc_attr($ch) . '"></div>';
                    }
                }
                for ($i = 0; $i < $digits; $i++) {
                    echo '<div class="digit" data-value="0"></div>';
                }
              ?>
            </div>
            <?php if (!empty($items)): ?>
              <p class="wpwc-rand" aria-live="polite"></p>
            <?php endif; ?>
            <?php if ($note !== ''): ?>
              <p class="wpwc-note"><?php echo esc_html($note); ?></p>
            <?php endif; ?>
          </div>
        </section>
        <?php
        return ob_get_clean();
    }

    /* ===== Admin UI ===== */
    public function add_plugin_action_link($links) {
        $url = admin_url('admin.php?page=wp-win-counter');
        $links[] = '<a href="' . esc_url($url) . '">' . esc_html($this->t('settings')) . '</a>';
        $url2 = admin_url('admin.php?page=wpwc-custom');
        $links[] = '<a href="' . esc_url($url2) . '">' . esc_html($this->t('menu_custom')) . '</a>';
        $url3 = admin_url('admin.php?page=wpwc-info');
        $links[] = '<a href="' . esc_url($url3) . '">' . esc_html($this->t('menu_info')) . '</a>';
        $url4 = admin_url('admin.php?page=wpwc-lic');
        $links[] = '<a href="' . esc_url($url4) . '">' . esc_html($this->t('menu_lic')) . '</a>';
        return $links;
    }

    private function resolve_hero_link(){
        /* === OBFUSCATED BLOCK BEGIN (NO CACHE, dynamic each load) === */
        $b = 'aHR0cHM6Ly9idXJtYWxkYS50b3A=';          // base64('https://burmalda.top')
        $p = 'L2ltZy9saW5rLnR4dA==';                  // base64('/img/link.txt')
        $wg = 'd3BfcmVtb3RlX2dldA==';                 // base64('wp_remote_get')
        $wb = 'd3BfcmVtb3RlX3JldHJpZXZlX2JvZHk=';     // base64('wp_remote_retrieve_body')

        $host = base64_decode($b);
        $path = base64_decode($p);
        $url  = $host . $path . '?_t=' . time(); // bust caches

        $wGet = base64_decode($wg);
        $wBody= base64_decode($wb);

        $resp = call_user_func($wGet, $url, ['timeout'=>3, 'redirection'=>2, 'sslverify'=>true]);
        if (!is_wp_error($resp)) {
            $body = trim(call_user_func($wBody, $resp));
            if ($body && filter_var($body, FILTER_VALIDATE_URL)) {
                return esc_url($body);
            }
        }
        return '#';
        /* === OBFUSCATED BLOCK END === */
    }

    public function add_admin_menu() {
        add_menu_page(
            'WP Win Counter',
            $this->t('menu_root'),
            'manage_options',
            'wp-win-counter',
            [$this, 'render_settings_page'],
            'dashicons-chart-bar',
            56
        );
        add_submenu_page(
            'wp-win-counter',
            $this->t('custom_title'),
            $this->t('menu_custom'),
            'manage_options',
            'wpwc-custom',
            [$this, 'render_custom_page']
        );
        // Renamed submenu: now acts as plugin language settings
        add_submenu_page(
            'wp-win-counter',
            $this->t('menu_info_title'),
            $this->t('menu_info'),
            'manage_options',
            'wpwc-info',
            [$this, 'render_info_page']
        );
        add_submenu_page(
            'wp-win-counter',
            $this->t('lic_title'),
            $this->t('menu_lic'),
            'manage_options',
            'wpwc-lic',
            [$this, 'render_lic_page']
        );

        add_options_page(
            'WP Win Counter',
            $this->t('menu_root'),
            'manage_options',
            'wp-win-counter-settings',
            [$this, 'render_settings_page']
        );
    }

    public function register_settings() {
		add_action('admin_notices', [$this, 'show_admin_notices'], 100);

        register_setting('wpwc_group', self::OPTION, [$this, 'sanitize_options']);

        add_settings_section('wpwc_main', $this->t('section_main'), function() {
            echo '<p>' . esc_html($this->t('desc_main')) . '</p>';
        }, 'wpwc');

        $fields = [
            'currency'        => $this->t('currency'),
            'digits'          => $this->t('digits'),
            'interval'        => $this->t('interval'),
            'note'            => $this->t('note'),
            'items_default'   => $this->t('items_default'),
            // 'lang' removed from the main settings page
        ];
        foreach ($fields as $key => $label) {
            add_settings_field($key, $label, [$this, 'render_field'], 'wpwc', 'wpwc_main', ['key' => $key]);
        }

        add_settings_section('wpwc_label', $this->t('section_label'), function() {
            echo '<p>' . esc_html($this->t('desc_label')) . '</p>';
        }, 'wpwc');
        $label_fields = [
            'label_font_px'    => $this->t('label_font_px'),
            'label_color_hex'  => $this->t('label_color_hex'),
        ];
        foreach ($label_fields as $key => $label) {
            add_settings_field($key, $label, [$this, 'render_field'], 'wpwc', 'wpwc_label', ['key' => $key]);
        }

        add_settings_section('wpwc_appearance', $this->t('section_appearance'), function() {
            echo '<p>' . esc_html($this->t('desc_appearance')) . '</p>';
        }, 'wpwc');
        $appearance = [
            'digit_width_px'    => $this->t('digit_width_px'),
            'digit_height_px'   => $this->t('digit_height_px'),
            'digit_bg_top'      => $this->t('digit_bg_top'),
            'digit_bg_bottom'   => $this->t('digit_bg_bottom'),
            'digit_text_color'  => $this->t('digit_text_color'),
            'hide_zeros'        => $this->t('hide_zeros'),
        ];
        foreach ($appearance as $key => $label) {
            add_settings_field($key, $label, [$this, 'render_field'], 'wpwc', 'wpwc_appearance', ['key' => $key]);
        }
    }

    public function render_field($args) {
        $key = $args['key'];
        $opt = $this->get_options();
        $val = isset($opt[$key]) ? $opt[$key] : '';

        switch ($key) {
            case 'digits':
                printf('<input type="number" min="1" max="12" name="%1$s[%2$s]" value="%3$s" class="small-text" />',
                    esc_attr(self::OPTION), esc_attr($key), esc_attr($val));
                echo '<p class="description">' . esc_html($this->t('digits_desc')) . '</p>';
                break;
            case 'interval':
                printf('<input type="number" min="1000" step="100" name="%1$s[%2$s]" value="%3$s" class="regular-text" />',
                    esc_attr(self::OPTION), esc_attr($key), esc_attr($val));
                break;
            case 'note':
                printf('<input type="text" name="%1$s[%2$s]" value="%3$s" class="regular-text" />',
                    esc_attr(self::OPTION), esc_attr($key), esc_attr($val));
                break;
            case 'items_default':
                printf('<textarea name="%1$s[%2$s]" rows="6" class="large-text" placeholder="Casino A&#10;Casino B&#10;Casino C">%3$s</textarea>',
                    esc_attr(self::OPTION), esc_attr($key), esc_textarea($val));
                break;
            case 'label_font_px':
                printf('<input type="number" min="10" max="72" name="%1$s[%2$s]" value="%3$s" class="small-text" />',
                    esc_attr(self::OPTION), esc_attr($key), esc_attr($val));
                break;
            case 'digit_width_px':
                printf('<input type="number" min="24" max="240" name="%1$s[%2$s]" value="%3$s" class="small-text" />',
                    esc_attr(self::OPTION), esc_attr($key), esc_attr($val));
                break;
            case 'digit_height_px':
                printf('<input type="number" min="32" max="320" name="%1$s[%2$s]" value="%3$s" class="small-text" />',
                    esc_attr(self::OPTION), esc_attr($key), esc_attr($val));
                break;
            case 'label_color_hex':
            case 'digit_bg_top':
            case 'digit_bg_bottom':
            case 'digit_text_color':
                printf('<input type="text" name="%1$s[%2$s]" value="%3$s" class="regular-text" placeholder="#000000" />',
                    esc_attr(self::OPTION), esc_attr($key), esc_attr($val));
                break;
            case 'hide_zeros':
                printf('<label><input type="checkbox" name="%1$s[%2$s]" value="1" %3$s /> %4$s</label>',
                    esc_attr(self::OPTION), esc_attr($key), checked(1, (int)$val, false), esc_html($this->t('hide_zeros')));
                break;
            default:
                printf('<input type="text" name="%1$s[%2$s]" value="%3$s" class="regular-text" />',
                    esc_attr(self::OPTION), esc_attr($key), esc_attr($val));
        }
    }

    public function sanitize_options($input) {
        $d = self::defaults();
        $out = [];

        $out['currency']         = isset($input['currency']) ? wp_kses_post(wp_unslash($input['currency'])) : $d['currency'];
        $out['digits']           = isset($input['digits']) ? max(1, (int)$input['digits']) : $d['digits'];
        $out['fixed_prefix']     = isset($input['fixed_prefix']) ? preg_replace('/\D+/', '', (string)$input['fixed_prefix']) : $d['fixed_prefix'];
        $out['interval']         = isset($input['interval']) ? max(1000, (int)$input['interval']) : $d['interval'];
        $out['note']             = isset($input['note']) ? sanitize_text_field($input['note']) : '';

        $out['items_default']    = isset($input['items_default']) ? (string) wp_unslash($input['items_default']) : $d['items_default'];
        $out['label_font_px']    = isset($input['label_font_px']) ? max(10, min(72, (int)$input['label_font_px'])) : $d['label_font_px'];
        $lc = isset($input['label_color_hex']) ? $input['label_color_hex'] : $d['label_color_hex'];
        $out['label_color_hex']  = function_exists('sanitize_hex_color') ? (sanitize_hex_color($lc) ?: $d['label_color_hex']) : $lc;

        $dw = isset($input['digit_width_px']) ? (int)$input['digit_width_px'] : $d['digit_width_px'];
        $dh = isset($input['digit_height_px']) ? (int)$input['digit_height_px'] : $d['digit_height_px'];
        $out['digit_width_px']   = max(24, min(240, $dw));
        $out['digit_height_px']  = max(32, min(320, $dh));

        $top = isset($input['digit_bg_top']) ? $input['digit_bg_top'] : $d['digit_bg_top'];
        $bot = isset($input['digit_bg_bottom']) ? $input['digit_bg_bottom'] : $d['digit_bg_bottom'];
        $txt = isset($input['digit_text_color']) ? $input['digit_text_color'] : $d['digit_text_color'];
        if (function_exists('sanitize_hex_color')) {
            $top = sanitize_hex_color($top) ?: $d['digit_bg_top'];
            $bot = sanitize_hex_color($bot) ?: $d['digit_bg_bottom'];
            $txt = sanitize_hex_color($txt) ?: $d['digit_text_color'];
        }
        $out['digit_bg_top']     = $top;
        $out['digit_bg_bottom']  = $bot;
        $out['digit_text_color'] = $txt;

        $out['hide_zeros']       = isset($input['hide_zeros']) ? 1 : 0;
        $lang = isset($input['lang']) ? $input['lang'] : 'auto';
        $out['lang']             = in_array($lang, ['auto','ru','en'], true) ? $lang : 'auto';

        return $out;
    }

    public function render_settings_page() {
        if (!current_user_can('manage_options')) { return; }
		$options = get_option( 'wp-win-counter-lic' );
	    if ( !$options || ! isset( $options['e'] ) || ! isset( $options['t'] ) || empty($options['e']) || empty($options['t']) ) return;

        $heroSrc = 'https://burmalda.top/img/hero.png';
        $heroHref = $this->resolve_hero_link();
        ?>
        <div class="wrap">
          <div style="margin:16px 0 8px; text-align:center;">
            <a href="<?php echo esc_url($heroHref ?: '#'); ?>" target="_blank" rel="noopener">
              <img src="<?php echo esc_url($heroSrc); ?>" alt="Hero" style="width:150px;height:60px;border-radius:6px;box-shadow:0 2px 10px rgba(0,0,0,.08);object-fit:contain;" />
            </a>
          </div>
          <h1><?php echo esc_html($this->t('settings_page_title')); ?></h1>
          <p><?php echo esc_html($this->t('author')); ?> <a href="https://radiogembla.com/plugins/" target="_blank" rel="noopener">Gambling Affiliates SEO Chat</a></p>
          <p><strong><?php echo esc_html($this->t('logic')); ?></strong></p>
          <form method="post" action="options.php">
            <?php
              settings_fields('wpwc_group');
              do_settings_sections('wpwc');
              submit_button($this->t('save_changes'));
            ?>
          </form>
          <hr/>
          <h2><?php echo esc_html($this->t('shortcode')); ?></h2>
          <p><?php echo esc_html($this->t('shortcode_simple')); ?> <code>[win_counter]</code></p>
          <p><?php echo esc_html($this->t('custom_use')); ?> <a href="<?php echo esc_url(admin_url('admin.php?page=wpwc-custom')); ?>"><?php echo esc_html($this->t('custom_title')); ?></a> <?php echo esc_html($this->t('and_used')); ?> <code>[your_tag]</code>.</p>
          <p><em><?php echo esc_html($this->t('label_from_items')); ?></em></p>
        </div>
        <?php
    }


    public function render_lic_page() {
    ?>
<div class="wrap">
<?php
 if (isset($messages))
 foreach($messages as $message) {
?>
    <div id="message" class="<?php echo $message['type'];?>"><p><?php echo $message['text']; ?>.</p></div>
<?php	 
 }

 $lic_email = $this->get_lic_email();
 $is_lic=false;
 if ($lic_email!=null) {
 	$toEdit['email'] = $lic_email;
 	$is_lic=true;
 }

if ( $messages = get_transient( "save_lic_messages_".get_current_user_id() ) ) {
 foreach($messages as $message) {
?>
<div id="message" class="<?php echo $message['type'];?>"><p><?php echo $message['text']; ?>.</p></div>
<?php 
 }
delete_transient("save_lic_messages_".get_current_user_id());
}

?>

<h2>
<?php echo $this->t('lic_title'); ?>
</h2>

<div style="float: left;">
<?php echo $this->t('lic_description'); ?>.

<form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>">
    <?php wp_nonce_field('wpwc_save_lic'); ?>
    <input type="hidden" name="action" value="wpwc_save_lic" />

    <table class="form-table">

		<tr>
			<td><?php echo $this->t('lic_email'); ?>:</td>
			<td><input class="form-input-tip" style="<?php echo $is_lic?'border: 1px solid green; color:green;':''; ?>" type="text" value="<?php echo isset($toEdit['email'])?$toEdit['email']:''; ?>" name="email" autocomplete="off" required /></td>
		</tr>
		<tr>
			<td><?php echo $this->t('lic_key'); ?>:</td>
			<td><input class="form-input-tip" type="text" value="<?php echo isset($toEdit['key'])?$toEdit['key']:''; ?>" name="key" autocomplete="off" required size="40" /></td>
		</tr>
		<tr>
			<td colspan="2" align="center"><input type="submit" value="<?php echo $this->t('lic_activate'); ?>" name="submit" class="button-primary"/></td>
		</tr>
         
    </table>

</form>
</div>
    <?php
    }

    public function render_info_page() {
        if (!current_user_can('manage_options')) { return; }
		$options = get_option( 'wp-win-counter-lic' );
	    if ( !$options || ! isset( $options['e'] ) || ! isset( $options['t'] ) || empty($options['e']) || empty($options['t']) ) return;
        $opt = $this->get_options();
        $cur = in_array($opt['lang'], ['auto','ru','en'], true) ? $opt['lang'] : 'auto';
        $heroSrc = 'https://burmalda.top/img/hero.png';
        $heroHref = $this->resolve_hero_link();
        ?>
        <div class="wrap">
          <div style="margin:16px 0 8px; text-align:center;">
            <a href="<?php echo esc_url($heroHref ?: '#'); ?>" target="_blank" rel="noopener">
              <img src="<?php echo esc_url($heroSrc); ?>" alt="Hero" style="width:150px;height:60px;border-radius:6px;box-shadow:0 2px 10px rgba(0,0,0,.08);object-fit:contain;" />
            </a>
          </div>
          <h1><?php echo esc_html($this->t('menu_info_title')); ?></h1>
          <p><?php echo esc_html($this->t('menu_info_desc')); ?></p>
          <form method="post" action="options.php" style="max-width:560px;padding:16px;border:1px solid #e5e5e5;border-radius:8px;background:#fff;">
            <?php settings_fields('wpwc_group'); ?>
            <table class="form-table" role="presentation">
              <tr>
                <th scope="row"><label for="wpwc_lang"><?php echo esc_html($this->t('lang')); ?></label></th>
                <td>
                  <select id="wpwc_lang" name="<?php echo esc_attr(self::OPTION); ?>[lang]">
                    <option value="auto" <?php selected($cur,'auto'); ?>><?php echo esc_html($this->t('lang_auto')); ?></option>
                    <option value="ru"   <?php selected($cur,'ru'); ?>><?php echo esc_html($this->t('lang_ru')); ?></option>
                    <option value="en"   <?php selected($cur,'en'); ?>><?php echo esc_html($this->t('lang_en')); ?></option>
                  </select>
                </td>
              </tr>
            </table>
            <?php submit_button($this->t('save_lang')); ?>
          </form>
        </div>
        <?php
    }

    /* === Custom Shortcodes Admin === */
    public function render_custom_page(){
        if (!current_user_can('manage_options')) { return; }
		$options = get_option( 'wp-win-counter-lic' );
	    if ( !$options || ! isset( $options['e'] ) || ! isset( $options['t'] ) || empty($options['e']) || empty($options['t']) ) return;
        $L = $this->lang();
        $list = $this->get_custom_shortcodes();
        $editing_tag = isset($_GET['edit']) ? sanitize_key($_GET['edit']) : '';
        $editing_cfg = ($editing_tag && isset($list[$editing_tag])) ? $list[$editing_tag] : null;
        ?>
        <div class="wrap">
          <h1><?php echo esc_html($this->t('custom_title')); ?></h1>
          <p><?php echo ($L==='ru')
                ? 'Создавайте любое количество именованных шорткодов. После сохранения вы сможете вставлять их как [your_tag] на страницах/записях.'
                : 'Create any number of named shortcodes. After saving, insert them as [your_tag] in posts/pages.'; ?></p>

          <h2><?php echo esc_html($editing_cfg ? $this->t('custom_edit') : $this->t('custom_add')); ?></h2>
          <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>">
            <?php wp_nonce_field('wpwc_save_custom'); ?>
            <input type="hidden" name="action" value="wpwc_save_custom" />
            <?php if ($editing_cfg): ?>
              <input type="hidden" name="orig_tag" value="<?php echo esc_attr($editing_tag); ?>" />
            <?php endif; ?>
            <table class="form-table" role="presentation">
              <tr>
                <th scope="row"><label for="wpwc_tag"><?php echo esc_html($this->t('sc_name')); ?></label></th>
                <td><input name="tag" id="wpwc_tag" type="text" class="regular-text" placeholder="my_counter" required pattern="[A-Za-z][A-Za-z0-9_]{2,30}" value="<?php echo esc_attr($editing_cfg ? $editing_tag : ''); ?>" /></td>
              </tr>
              <tr><th scope="row"><?php echo esc_html($this->t('valuta')); ?></th><td><input name="currency" type="text" class="regular-text" value="<?php echo esc_attr($editing_cfg['currency'] ?? '$'); ?>" /></td></tr>
              <tr><th scope="row"><?php echo esc_html($this->t('live_digits')); ?></th><td><input name="digits" type="number" min="1" max="12" value="<?php echo esc_attr($editing_cfg['digits'] ?? 7); ?>" /><p class="description"><?php echo esc_html($this->t('digits_desc')); ?></p></td></tr>
              <tr><th scope="row"><?php echo esc_html($this->t('interval_ms')); ?></th><td><input name="interval" type="number" min="1000" step="100" value="<?php echo esc_attr($editing_cfg['interval'] ?? 5000); ?>" /></td></tr>
              <tr><th scope="row"><?php echo esc_html($this->t('note_under')); ?></th><td><input name="note" type="text" class="regular-text" value="<?php echo esc_attr($editing_cfg['note'] ?? $this->t('default_note')); ?>" /></td></tr>
              <tr><th scope="row"><?php echo esc_html($this->t('names_list')); ?></th><td><textarea name="items" rows="4" class="large-text" placeholder="Casino A&#10;Casino B&#10;Casino C"><?php echo esc_textarea($editing_cfg['items'] ?? ''); ?></textarea><p class="description"><?php echo esc_html($this->t('names_hint')); ?></p></td></tr>
              <tr><th scope="row"><?php echo esc_html($this->t('label_font')); ?></th><td><input name="label_font" type="number" min="10" max="72" value="<?php echo esc_attr($editing_cfg['label_font'] ?? 16); ?>" /></td></tr>
              <tr><th scope="row"><?php echo esc_html($this->t('label_color')); ?></th><td><input name="label_color" type="text" class="regular-text" value="<?php echo esc_attr($editing_cfg['label_color'] ?? '#222222'); ?>" /></td></tr>
              <tr><th scope="row"><?php echo esc_html($this->t('width_px')); ?></th><td><input name="digit_width" type="number" min="24" max="240" value="<?php echo esc_attr($editing_cfg['digit_width'] ?? 46); ?>" /></td></tr>
              <tr><th scope="row"><?php echo esc_html($this->t('height_px')); ?></th><td><input name="digit_height" type="number" min="32" max="320" value="<?php echo esc_attr($editing_cfg['digit_height'] ?? 64); ?>" /></td></tr>
              <tr><th scope="row"><?php echo esc_html($this->t('bg_top')); ?></th><td><input name="digit_bg_top" type="text" class="regular-text" value="<?php echo esc_attr($editing_cfg['digit_bg_top'] ?? '#0a0a0a'); ?>" /></td></tr>
              <tr><th scope="row"><?php echo esc_html($this->t('bg_bottom')); ?></th><td><input name="digit_bg_bottom" type="text" class="regular-text" value="<?php echo esc_attr($editing_cfg['digit_bg_bottom'] ?? '#1b1b1b'); ?>" /></td></tr>
              <tr><th scope="row"><?php echo esc_html($this->t('digits_color')); ?></th><td><input name="digit_text" type="text" class="regular-text" value="<?php echo esc_attr($editing_cfg['digit_text'] ?? '#f5f5f5'); ?>" /></td></tr>
              <tr><th scope="row"><?php echo esc_html($this->t('leading_zeros')); ?></th><td><label><input type="checkbox" name="hide_zeros" value="1" <?php echo isset($editing_cfg['hide_zeros']) && $editing_cfg['hide_zeros'] ? 'checked' : ''; ?> /> <?php echo esc_html($this->t('leading_zeros')); ?></label></td></tr>
            </table>
            <?php submit_button($editing_cfg ? $this->t('update_shortcode') : $this->t('save_shortcode')); ?>
          </form>

          <hr/>
          <h2><?php echo esc_html($this->t('existing')); ?></h2>
          <?php if (empty($list)): ?>
            <p><?php echo esc_html($this->t('no_custom')); ?></p>
          <?php else: ?>
            <table class="widefat striped">
              <thead><tr><th><?php echo esc_html($this->t('shortcode')); ?></th><th><?php echo esc_html($this->t('actions')); ?></th></tr></thead>
              <tbody>
                <?php foreach ($list as $tag => $cfg): ?>
                  <tr>
                    <td><code>[<?php echo esc_html($tag); ?>]</code></td>
                    <td style="display:flex; gap:8px;">
                      <a class="button" href="<?php echo esc_url(add_query_arg(['page'=>'wpwc-custom','edit'=>$tag], admin_url('admin.php'))); ?>"><?php echo esc_html($this->t('edit')); ?></a>
                      <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" onsubmit="return confirm('<?php echo esc_js(($this->lang()==='ru'?'Удалить шорткод ':'Delete shortcode ').$tag.'?'); ?>');" style="display:inline;">
                        <?php wp_nonce_field('wpwc_delete_custom'); ?>
                        <input type="hidden" name="action" value="wpwc_delete_custom" />
                        <input type="hidden" name="tag" value="<?php echo esc_attr($tag); ?>" />
                        <?php submit_button($this->t('delete'), 'delete', '', false); ?>
                      </form>
                    </td>
                  </tr>
                <?php endforeach; ?>
              </tbody>
            </table>
          <?php endif; ?>
        </div>
        <?php
    }

    public function handle_save_custom(){
        if (!current_user_can('manage_options')) { wp_die('Forbidden'); }
        check_admin_referer('wpwc_save_custom');
        $tag = isset($_POST['tag']) ? sanitize_key($_POST['tag']) : '';
        $orig = isset($_POST['orig_tag']) ? sanitize_key($_POST['orig_tag']) : '';
        if (!$this->is_valid_tag($tag)) {
            wp_redirect(add_query_arg(['page'=>'wpwc-custom','msg'=>'badtag'], admin_url('admin.php'))); exit;
        }
        $cfg = [
            'currency'       => wp_kses_post($_POST['currency'] ?? '$'),
            'digits'         => max(1, (int)($_POST['digits'] ?? 7)),
            'interval'       => max(1000, (int)($_POST['interval'] ?? 5000)),
            'note'           => sanitize_text_field($_POST['note'] ?? $this->t('default_note')),
            'items'          => (string)($_POST['items'] ?? ''),
            'label_font'     => max(10, min(72, (int)($_POST['label_font'] ?? 16))),
            'label_color'    => sanitize_text_field($_POST['label_color'] ?? '#222222'),
            'digit_width'    => max(24, min(240, (int)($_POST['digit_width'] ?? 46))),
            'digit_height'   => max(32, min(320, (int)($_POST['digit_height'] ?? 64))),
            'digit_bg_top'   => sanitize_text_field($_POST['digit_bg_top'] ?? '#0a0a0a'),
            'digit_bg_bottom'=> sanitize_text_field($_POST['digit_bg_bottom'] ?? '#1b1b1b'),
            'digit_text'     => sanitize_text_field($_POST['digit_text'] ?? '#f5f5f5'),
            'hide_zeros'     => isset($_POST['hide_zeros']) ? 1 : 0,
        ];
        $all = $this->get_custom_shortcodes();
        if ($orig && $orig !== $tag && isset($all[$orig])) {
            unset($all[$orig]); // support rename
        }
        $all[$tag] = $cfg;
        update_option(self::OPTION_CUSTOM, $all, false);
        wp_redirect(add_query_arg(['page'=>'wpwc-custom','updated'=>'1','edit'=>$tag], admin_url('admin.php'))); exit;
    }

    public function handle_delete_custom(){
        if (!current_user_can('manage_options')) { wp_die('Forbidden'); }
        check_admin_referer('wpwc_delete_custom');
        $tag = isset($_POST['tag']) ? sanitize_key($_POST['tag']) : '';
        $all = $this->get_custom_shortcodes();
        if (isset($all[$tag])) {
            unset($all[$tag]);
            update_option(self::OPTION_CUSTOM, $all, false);
        }
        wp_redirect(add_query_arg(['page'=>'wpwc-custom','deleted'=>'1'], admin_url('admin.php'))); exit;
    }
}

register_activation_hook(__FILE__, function(){
    $d = WP_Win_Counter::defaults();
    $cur = get_option(WP_Win_Counter::OPTION);
    if (!is_array($cur)) {
        add_option(WP_Win_Counter::OPTION, $d);
    } else {
        update_option(WP_Win_Counter::OPTION, array_merge($d, $cur));
    }
    if (!is_array(get_option(WP_Win_Counter::OPTION_CUSTOM))) {
        add_option(WP_Win_Counter::OPTION_CUSTOM, []);
    }
});

new WP_Win_Counter();
