(function(){
  // ---------- Helpers ----------
  function padN(n, count){
    const s = String(Math.max(0, Math.floor(n)));
    if (s.length >= count) return s.slice(-count);
    return (('0'.repeat(count)) + s).slice(-count);
  }

  function buildReels(counter){
    const digits = counter.querySelectorAll('.digit');
    digits.forEach(d=>{
      const reel = document.createElement('div');
      reel.className = 'reel';

      if(d.classList.contains('fixed')){
        const cell = document.createElement('div');
        cell.className = 'cell';
        cell.textContent = d.dataset.value || '0';
        reel.appendChild(cell);
      } else {
        for(let i=0;i<10;i++){
          const cell = document.createElement('div');
          cell.className = 'cell';
          cell.textContent = i;
          reel.appendChild(cell);
        }
      }
      d.appendChild(reel);
    });
  }

  function getSlotHeight(counter){
    const h = parseInt(counter.getAttribute('data-digit-height') || '64', 10);
    return isNaN(h) ? 64 : h;
  }

  function setNumber(counter, n, animate=true){
    const SLOT_HEIGHT = getSlotHeight(counter);
    const liveCount = parseInt(counter.dataset.digits || '7', 10);
    const hideZeros = counter.getAttribute('data-hide-zeros') === '1';
    const str = padN(n, liveCount);
    const rolling = [...counter.querySelectorAll('.digit:not(.fixed)')];

    [...str].forEach((ch,i)=>{
      const val  = parseInt(ch,10);
      const reel = rolling[i]?.querySelector('.reel');
      if(!reel) return;
      reel.style.transitionDuration = animate ? (650 + Math.random()*500) + 'ms' : '0ms';
      reel.style.transform = `translateY(${-val*SLOT_HEIGHT}px)`;
      rolling[i].dataset.value = String(val);
    });

    // Leading zeros visibility
    let nonZeroSeen = false;
    for(let i=0;i<rolling.length;i++){
      const ch = str[i];
      const isLast = i === rolling.length - 1;
      if(hideZeros && !nonZeroSeen && ch === '0' && !isLast){
        rolling[i].classList.add('dim');
      } else {
        nonZeroSeen = nonZeroSeen || ch !== '0' || isLast;
        rolling[i].classList.remove('dim');
      }
    }
  }

  function pickRandomDifferent(items, prevIndex){
    if(!items || !items.length) return {index:-1, value:''};
    if(items.length === 1) return {index:0, value:items[0]};
    let idx;
    do { idx = Math.floor(Math.random()*items.length); } while(idx === prevIndex);
    return {index: idx, value: items[idx]};
  }

  function setRandomLabel(counter, labelEl){
    if(!labelEl) return;
    const json = counter.getAttribute('data-items');
    if(!json) return;
    let items = [];
    try { items = JSON.parse(json); } catch(e){ return; }
    if(!Array.isArray(items) || !items.length) return;

    const prev = parseInt(labelEl.getAttribute('data-prev-index') || '-1', 10);
    const {index, value} = pickRandomDifferent(items, isNaN(prev) ? -1 : prev);
    labelEl.textContent = value;
    labelEl.setAttribute('data-prev-index', String(index));
  }

  // ---------- Keys for scheduling ----------
  function hourKey(d){ return `${d.getFullYear()}-${d.getMonth()}-${d.getDate()}-${d.getHours()}`; }
  function ninetyKey(d){ // id of current 90-minute window
    const t = d.getTime();
    const win = Math.floor(t / (90*60*1000));
    return `n90-${win}`;
  }

  // ---------- Plans ----------
  function buildPlanHour(ticksPerHour){
    const perHalf = Math.max(1, Math.floor(ticksPerHour / 2));
    const seven = Math.floor(Math.random()*ticksPerHour);
    let six1 = Math.floor(Math.random()*perHalf);
    let six2 = Math.floor(Math.random()*perHalf) + perHalf;
    if (six1 === seven) six1 = (six1 + 1) % perHalf;
    if (six2 === seven) six2 = ((six2 - perHalf + 1) % perHalf) + perHalf;
    return { sevenTick: seven, sixTicks: [six1, six2], ticksPerHour };
  }
  function getPlanHour(ticksPerHour){
    try {
      const now = new Date();
      const key = `wpwc_plan_${hourKey(now)}_${ticksPerHour}_v196`;
      const raw = localStorage.getItem(key);
      if(raw){
        const obj = JSON.parse(raw);
        if(obj && typeof obj.sevenTick === 'number' && Array.isArray(obj.sixTicks) && obj.ticksPerHour === ticksPerHour){
          return {key, ...obj};
        }
      }
      const plan = buildPlanHour(ticksPerHour);
      localStorage.setItem(key, JSON.stringify(plan));
      return {key, ...plan};
    } catch(e){
      return buildPlanHour(ticksPerHour);
    }
  }

  function buildPlan90(ticksPer90){
    const eight = Math.floor(Math.random()*ticksPer90);
    return { eightTick: eight, ticksPer90 };
  }
  function getPlan90(ticksPer90){
    try {
      const now = new Date();
      const key = `wpwc_plan90_${ninetyKey(now)}_${ticksPer90}_v196`;
      const raw = localStorage.getItem(key);
      if(raw){
        const obj = JSON.parse(raw);
        if(obj && typeof obj.eightTick === 'number' && obj.ticksPer90 === ticksPer90){
          return {key, ...obj};
        }
      }
      const plan = buildPlan90(ticksPer90);
      localStorage.setItem(key, JSON.stringify(plan));
      return {key, ...plan};
    } catch(e){
      return buildPlan90(ticksPer90);
    }
  }

  // ---------- Ticks ----------
  function getTickIndexHour(intervalMs){
    const now = new Date();
    const s = now.getMinutes()*60 + now.getSeconds(); // 0..3599 within hour
    return Math.floor(s / (intervalMs/1000));
  }
  function getTickIndex90(intervalMs){
    const now = new Date();
    const sinceMidnight = now.getHours()*3600 + now.getMinutes()*60 + now.getSeconds();
    const within90 = sinceMidnight % 5400; // seconds within 90-min window
    return Math.floor(within90 / (intervalMs/1000));
  }

  // ---------- Value generation ----------
  function valueForMode(mode){
    switch(mode){
      case '8': return 10000000 + Math.floor(Math.random()*10000000); // 10,000,000 — 19,999,999
      case '7': return 1000000  + Math.floor(Math.random()*1000000);  // 1,000,000 — 1,999,999
      case '6': return 100000   + Math.floor(Math.random()*900000);
      case '5': return 10000    + Math.floor(Math.random()*90000);
      case '4': return 1000     + Math.floor(Math.random()*9000);
      case '3': return 100      + Math.floor(Math.random()*900);
      default:  return 0;
    }
  }

  function autoscale(counter){
    const wrap = counter.closest('.wpwc-center');
    if(!wrap) return;
    counter.style.transform = 'scale(1)';
    const available = wrap.clientWidth - 4;
    const needed = counter.scrollWidth;
    const scale = needed > 0 ? Math.min(1, available / needed) : 1;
    counter.style.transform = `scale(${scale})`;
  }

  function initOne(counter){
    if(counter.dataset.initialized === '1') return;
    counter.dataset.initialized = '1';

    buildReels(counter);

    const interval = Math.max(1000, parseInt(counter.dataset.interval || '5000', 10));
    const ticksPerHour = Math.max(1, Math.floor(3600 / (interval/1000)));
    const ticksPer90   = Math.max(1, Math.floor(5400 / (interval/1000)));
    const planHour = getPlanHour(ticksPerHour);
    const plan90   = getPlan90(ticksPer90);

    function tick(updateLabel){
      const liveCount = parseInt(counter.dataset.digits || '7', 10);
      const idxHour = getTickIndexHour(interval) % ticksPerHour;
      const idx90   = getTickIndex90(interval) % ticksPer90;

      let mode;
      if (liveCount >= 8 && idx90 === plan90.eightTick) {
        mode = '8';
      } else if (idxHour === planHour.sevenTick) {
        mode = '7';
      } else if (planHour.sixTicks.includes(idxHour)) {
        mode = '6';
      } else {
        const r = idxHour % 3;
        mode = r === 0 ? '3' : r === 1 ? '4' : '5';
      }

      const n = valueForMode(mode);
      setNumber(counter, n, true);

      if(updateLabel){
        const labelEl = counter.parentElement.querySelector('.wpwc-rand');
        setRandomLabel(counter, labelEl);
      }
      autoscale(counter);
    }

    tick(true);
    setInterval(()=> tick(true), interval);
    window.addEventListener('resize', ()=>autoscale(counter));
  }

  function initAll(){
    document.querySelectorAll('.win-counter').forEach(initOne);
  }

  if(document.readyState === 'loading'){
    document.addEventListener('DOMContentLoaded', initAll);
  } else {
    initAll();
  }
})();